<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * Модель для таблицы "games"
 *
 * @property int $id
 * @property string $name
 * @property string $category
 * @property string|null $description
 * @property int|null $min_players
 * @property int|null $max_players
 * @property int|null $duration
 * @property string|null $image
 * @property int|null $is_popular
 * @property int|null $created_at
 * @property int|null $updated_at
 *
 * @property GameSession[] $sessions
 */
class Game extends ActiveRecord
{
    const CATEGORY_CLASSIC = 'классические';
    const CATEGORY_CARD = 'карточные';
    const CATEGORY_ECONOMIC = 'экономические';
    const CATEGORY_STRATEGY = 'стратегии';
    const CATEGORY_CHILDREN = 'для детей';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%games}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['name', 'category'], 'required'],
            [['category'], 'string'],
            [['min_players', 'max_players', 'duration', 'is_popular', 'created_at', 'updated_at'], 'integer'],
            [['name', 'image'], 'string', 'max' => 255],
            [['description'], 'string'],
            [['name'], 'unique', 'message' => 'Игра с таким названием уже существует'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'name' => 'Название игры',
            'category' => 'Категория',
            'description' => 'Описание',
            'min_players' => 'Минимум игроков',
            'max_players' => 'Максимум игроков',
            'duration' => 'Длительность (мин)',
            'image' => 'Изображение',
            'is_popular' => 'Популярная игра',
            'created_at' => 'Дата добавления',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * Получить все игровые сессии для этой игры
     */
    public function getSessions()
    {
        return $this->hasMany(GameSession::class, ['game_id' => 'id']);
    }

    /**
     * Получить активные сессии
     */
    public function getActiveSessions()
    {
        return $this->getSessions()
            ->where(['>=', 'date', date('Y-m-d')])
            ->andWhere(['status' => GameSession::STATUS_PLANNED])
            ->orderBy(['date' => SORT_ASC, 'time' => SORT_ASC]);
    }

    /**
     * Получить список всех категорий
     */
    public static function getCategoriesList()
    {
        return [
            self::CATEGORY_CLASSIC => 'Классические',
            self::CATEGORY_CARD => 'Карточные',
            self::CATEGORY_ECONOMIC => 'Экономические',
            self::CATEGORY_STRATEGY => 'Стратегии',
            self::CATEGORY_CHILDREN => 'Для детей',
        ];
    }

    /**
     * Получить название категории
     */
    public function getCategoryLabel()
    {
        $list = self::getCategoriesList();
        return $list[$this->category] ?? $this->category;
    }

    /**
     * Получить популярные игры
     */
    public static function getPopularGames($limit = 6)
    {
        return self::find()
            ->where(['is_popular' => true])
            ->limit($limit)
            ->all();
    }

    /**
     * Получить игры по категории
     */
    public static function getGamesByCategory($category)
    {
        return self::find()
            ->where(['category' => $category])
            ->orderBy(['name' => SORT_ASC])
            ->all();
    }

    /**
     * Получить все категории с количеством игр
     */
    public static function getCategoriesWithCount()
    {
        $categories = self::find()
            ->select(['category', 'COUNT(*) as count'])
            ->groupBy(['category'])
            ->orderBy(['category' => SORT_ASC])
            ->asArray()
            ->all();
            
        $result = [];
        $list = self::getCategoriesList();
        
        foreach ($categories as $category) {
            $result[$category['category']] = [
                'name' => $list[$category['category']] ?? $category['category'],
                'count' => $category['count']
            ];
        }
        
        return $result;
    }
}