<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * Модель для таблицы "game_sessions"
 *
 * @property int $id
 * @property int $game_id
 * @property string $date
 * @property string $time
 * @property string $address
 * @property float $price
 * @property int $max_players
 * @property int $current_players
 * @property string $status
 * @property int|null $created_at
 * @property int|null $updated_at
 *
 * @property Game $game
 * @property Request[] $requests
 */
class GameSession extends ActiveRecord
{
    const STATUS_PLANNED = 'planned';
    const STATUS_ACTIVE = 'active';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%game_sessions}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['game_id', 'date', 'time', 'address', 'price', 'max_players'], 'required'],
            [['game_id', 'max_players', 'current_players', 'created_at', 'updated_at'], 'integer'],
            [['price'], 'number'],
            [['date', 'time'], 'safe'],
            [['status'], 'string'],
            [['address'], 'string', 'max' => 500],
            [['game_id'], 'exist', 'skipOnError' => true, 'targetClass' => Game::class, 'targetAttribute' => ['game_id' => 'id']],
            ['current_players', 'default', 'value' => 0],
            ['status', 'default', 'value' => self::STATUS_PLANNED],
            ['current_players', 'compare', 'compareAttribute' => 'max_players', 'operator' => '<=', 'type' => 'number', 'message' => 'Количество игроков не может превышать максимальное'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'game_id' => 'Игра',
            'date' => 'Дата',
            'time' => 'Время',
            'address' => 'Адрес',
            'price' => 'Стоимость',
            'max_players' => 'Макс. игроков',
            'current_players' => 'Записано',
            'status' => 'Статус',
            'gameName' => 'Игра',
            'fullDatetime' => 'Дата и время',
            'availableSlots' => 'Свободно мест',
        ];
    }

    /**
     * Получить игру
     */
    public function getGame()
    {
        return $this->hasOne(Game::class, ['id' => 'game_id']);
    }

    /**
     * Получить заявки на эту сессию
     */
    public function getRequests()
    {
        return $this->hasMany(Request::class, ['session_id' => 'id']);
    }

    /**
     * Получить подтвержденные заявки
     */
    public function getApprovedRequests()
    {
        return $this->getRequests()->where(['status' => Request::STATUS_APPROVED]);
    }

    /**
     * Получить название игры
     */
    public function getGameName()
    {
        return $this->game ? $this->game->name : 'Неизвестная игра';
    }

    /**
     * Получить полную дату и время
     */
    public function getFullDatetime()
    {
        return date('d.m.Y', strtotime($this->date)) . ' в ' . date('H:i', strtotime($this->time));
    }

    /**
     * Получить доступные места
     */
    public function getAvailableSlots()
    {
        return $this->max_players - $this->current_players;
    }

    /**
     * Проверить доступность
     */
    public function isAvailable()
    {
        return $this->availableSlots > 0 && $this->status === self::STATUS_PLANNED;
    }

    /**
     * Получить статус на русском
     */
    public function getStatusLabel()
    {
        $statuses = [
            self::STATUS_PLANNED => 'Запланирована',
            self::STATUS_ACTIVE => 'Идет набор',
            self::STATUS_COMPLETED => 'Завершена',
            self::STATUS_CANCELLED => 'Отменена',
        ];
        
        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * Получить цвет статуса
     */
    public function getStatusColor()
    {
        $colors = [
            self::STATUS_PLANNED => 'warning',
            self::STATUS_ACTIVE => 'success',
            self::STATUS_COMPLETED => 'secondary',
            self::STATUS_CANCELLED => 'danger',
        ];
        
        return $colors[$this->status] ?? 'light';
    }

    /**
     * Получить сессии на ближайший месяц
     */
    public static function getForNextMonth($limit = 20)
    {
        return self::find()
            ->with('game')
            ->where(['>=', 'date', date('Y-m-d')])
            ->andWhere(['<=', 'date', date('Y-m-d', strtotime('+30 days'))])
            ->andWhere(['status' => self::STATUS_PLANNED])
            ->orderBy(['date' => SORT_ASC, 'time' => SORT_ASC])
            ->limit($limit)
            ->all();
    }

    /**
     * Увеличить количество игроков
     */
    public function incrementPlayers($count = 1)
    {
        $this->current_players += $count;
        return $this->save();
    }
}