<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * Модель для таблицы "requests"
 *
 * @property int $id
 * @property int $user_id
 * @property int $session_id
 * @property string $payment_method
 * @property string $status
 * @property int $player_count
 * @property float $total_price
 * @property int $consent_data
 * @property string|null $admin_comment
 * @property int $created_at
 * @property int $updated_at
 *
 * @property User $user
 * @property GameSession $session
 */
class Request extends ActiveRecord
{
    const STATUS_NEW = 'новая';
    const STATUS_APPROVED = 'одобрена';
    const STATUS_COMPLETED = 'завершена';
    const STATUS_CANCELLED = 'отменена';

    const PAYMENT_CASH = 'наличными';
    const PAYMENT_CARD = 'картой';
    const PAYMENT_TRANSFER = 'переводом';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%requests}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['user_id', 'session_id', 'payment_method', 'player_count', 'total_price', 'consent_data'], 'required'],
            [['user_id', 'session_id', 'player_count', 'consent_data', 'created_at', 'updated_at'], 'integer'],
            [['total_price'], 'number'],
            [['payment_method', 'status'], 'string'],
            [['admin_comment'], 'string'],
            ['consent_data', 'compare', 'compareValue' => 1, 'message' => 'Необходимо согласие на обработку персональных данных'],
            ['player_count', 'integer', 'min' => 1, 'max' => 5, 'message' => 'Количество игроков должно быть от 1 до 5'],
            ['status', 'default', 'value' => self::STATUS_NEW],
            [['session_id'], 'exist', 'skipOnError' => true, 'targetClass' => GameSession::class, 'targetAttribute' => ['session_id' => 'id']],
            [['user_id'], 'exist', 'skipOnError' => true, 'targetClass' => User::class, 'targetAttribute' => ['user_id' => 'id']],
            [['user_id', 'session_id'], 'unique', 'targetAttribute' => ['user_id', 'session_id'], 'message' => 'Вы уже подавали заявку на эту игру'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => '№ заявки',
            'user_id' => 'Пользователь',
            'session_id' => 'Игровая сессия',
            'payment_method' => 'Способ оплаты',
            'status' => 'Статус',
            'player_count' => 'Количество игроков',
            'total_price' => 'Сумма',
            'consent_data' => 'Согласие на обработку данных',
            'admin_comment' => 'Комментарий администратора',
            'created_at' => 'Дата подачи',
            'updated_at' => 'Дата обновления',
            'userName' => 'ФИО',
            'userPhone' => 'Телефон',
            'userEmail' => 'Email',
            'gameName' => 'Игра',
            'sessionDate' => 'Дата игры',
            'sessionTime' => 'Время',
            'sessionAddress' => 'Адрес',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($this->isNewRecord) {
                $this->created_at = time();
                
                // Увеличиваем счетчик игроков в сессии
                $session = GameSession::findOne($this->session_id);
                if ($session) {
                    $session->current_players += $this->player_count;
                    $session->save();
                }
            }
            $this->updated_at = time();
            return true;
        }
        return false;
    }

    /**
     * Получить пользователя
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * Получить игровую сессию
     */
    public function getSession()
    {
        return $this->hasOne(GameSession::class, ['id' => 'session_id']);
    }

    /**
     * Получить имя пользователя
     */
    public function getUserName()
    {
        return $this->user ? $this->user->full_name : 'Неизвестно';
    }

    /**
     * Получить телефон пользователя
     */
    public function getUserPhone()
    {
        return $this->user ? $this->user->phone : 'Неизвестно';
    }

    /**
     * Получить email пользователя
     */
    public function getUserEmail()
    {
        return $this->user ? $this->user->email : 'Неизвестно';
    }

    /**
     * Получить название игры
     */
    public function getGameName()
    {
        return $this->session && $this->session->game ? $this->session->game->name : 'Неизвестная игра';
    }

    /**
     * Получить дату игры
     */
    public function getSessionDate()
    {
        return $this->session ? date('d.m.Y', strtotime($this->session->date)) : 'Неизвестно';
    }

    /**
     * Получить время игры
     */
    public function getSessionTime()
    {
        return $this->session ? date('H:i', strtotime($this->session->time)) : 'Неизвестно';
    }

    /**
     * Получить адрес игры
     */
    public function getSessionAddress()
    {
        return $this->session ? $this->session->address : 'Неизвестно';
    }

    /**
     * Получить статус на русском
     */
    public function getStatusLabel()
    {
        $statuses = [
            self::STATUS_NEW => 'Новая',
            self::STATUS_APPROVED => 'Одобрена',
            self::STATUS_COMPLETED => 'Завершена',
            self::STATUS_CANCELLED => 'Отменена',
        ];
        
        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * Получить цвет статуса
     */
    public function getStatusColor()
    {
        $colors = [
            self::STATUS_NEW => 'warning',
            self::STATUS_APPROVED => 'success',
            self::STATUS_COMPLETED => 'primary',
            self::STATUS_CANCELLED => 'danger',
        ];
        
        return $colors[$this->status] ?? 'light';
    }

    /**
     * Получить способы оплаты
     */
    public static function getPaymentMethods()
    {
        return [
            self::PAYMENT_CASH => 'Наличными',
            self::PAYMENT_CARD => 'Картой',
            self::PAYMENT_TRANSFER => 'Переводом',
        ];
    }

    /**
     * Получить метод оплаты на русском
     */
    public function getPaymentMethodLabel()
    {
        $methods = self::getPaymentMethods();
        return $methods[$this->payment_method] ?? $this->payment_method;
    }

    /**
     * Получить статусы для админки
     */
    public static function getStatuses()
    {
        return [
            self::STATUS_NEW => 'Новая',
            self::STATUS_APPROVED => 'Одобрена',
            self::STATUS_COMPLETED => 'Завершена',
            self::STATUS_CANCELLED => 'Отменена',
        ];
    }

    /**
     * Получить заявки пользователя
     */
    public static function getUserRequests($userId)
    {
        return self::find()
            ->with(['session.game'])
            ->where(['user_id' => $userId])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();
    }

    /**
     * Получить новые заявки для админа
     */
    public static function getNewRequests()
    {
        return self::find()
            ->with(['user', 'session.game'])
            ->where(['status' => self::STATUS_NEW])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();
    }
}