<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * Модель для таблицы "reviews"
 *
 * @property int $id
 * @property int $user_id
 * @property int $session_id
 * @property int $rating
 * @property string $comment
 * @property string $status
 * @property int $created_at
 * @property int $updated_at
 *
 * @property User $user
 * @property GameSession $session
 */
class Review extends ActiveRecord
{
    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%reviews}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['user_id', 'session_id', 'rating', 'comment'], 'required'],
            [['user_id', 'session_id', 'rating', 'created_at', 'updated_at'], 'integer'],
            [['comment'], 'string'],
            [['status'], 'string'],
            ['rating', 'integer', 'min' => 1, 'max' => 5, 'message' => 'Оценка должна быть от 1 до 5'],
            ['status', 'default', 'value' => self::STATUS_PENDING],
            [['user_id'], 'exist', 'skipOnError' => true, 'targetClass' => User::class, 'targetAttribute' => ['user_id' => 'id']],
            [['session_id'], 'exist', 'skipOnError' => true, 'targetClass' => GameSession::class, 'targetAttribute' => ['session_id' => 'id']],
            [['user_id', 'session_id'], 'unique', 'targetAttribute' => ['user_id', 'session_id'], 'message' => 'Вы уже оставляли отзыв на эту игру'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'Пользователь',
            'session_id' => 'Игровая сессия',
            'rating' => 'Оценка',
            'comment' => 'Комментарий',
            'status' => 'Статус',
            'userName' => 'Пользователь',
            'gameName' => 'Игра',
            'created_at' => 'Дата отзыва',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($this->isNewRecord) {
                $this->created_at = time();
            }
            $this->updated_at = time();
            return true;
        }
        return false;
    }

    /**
     * Получить пользователя
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * Получить игровую сессию
     */
    public function getSession()
    {
        return $this->hasOne(GameSession::class, ['id' => 'session_id']);
    }

    /**
     * Получить имя пользователя
     */
    public function getUserName()
    {
        return $this->user ? $this->user->full_name : 'Аноним';
    }

    /**
     * Получить название игры
     */
    public function getGameName()
    {
        return $this->session && $this->session->game ? $this->session->game->name : 'Неизвестная игра';
    }

    /**
     * Получить статус на русском
     */
    public function getStatusLabel()
    {
        $statuses = [
            self::STATUS_PENDING => 'На модерации',
            self::STATUS_APPROVED => 'Опубликован',
            self::STATUS_REJECTED => 'Отклонен',
        ];
        
        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * Получить цвет статуса
     */
    public function getStatusColor()
    {
        $colors = [
            self::STATUS_PENDING => 'warning',
            self::STATUS_APPROVED => 'success',
            self::STATUS_REJECTED => 'danger',
        ];
        
        return $colors[$this->status] ?? 'light';
    }

    /**
     * Получить звезды для отображения рейтинга
     */
    public function getRatingStars()
    {
        $stars = '';
        for ($i = 1; $i <= 5; $i++) {
            if ($i <= $this->rating) {
                $stars .= '<i class="fas fa-star text-warning"></i>';
            } else {
                $stars .= '<i class="far fa-star text-warning"></i>';
            }
        }
        return $stars;
    }

    /**
     * Получить одобренные отзывы
     */
    public static function getApprovedReviews($limit = 10)
    {
        return self::find()
            ->with(['user', 'session.game'])
            ->where(['status' => self::STATUS_APPROVED])
            ->orderBy(['created_at' => SORT_DESC])
            ->limit($limit)
            ->all();
    }

    /**
     * Получить средний рейтинг для игры
     */
    public static function getAverageRatingForGame($gameId)
    {
        return self::find()
            ->joinWith('session')
            ->where(['game_sessions.game_id' => $gameId, 'reviews.status' => self::STATUS_APPROVED])
            ->average('rating');
    }
}